unit libsamplerate;

{$mode objfpc}{$H+}

// -----------------------------------------------------------------------------
// samplerate.h
//
// Copyright (c) 2002-2016, Erik de Castro Lopo <erikd@mega-nerd.com>
// All rights reserved.
//
// This code is released under 2-clause BSD license. Please see the
// file at : https://github.com/erikd/libsamplerate/blob/master/COPYING
//
// API documentation is available here:    http://www.mega-nerd.com/SRC/api.html
//
// -----------------------------------------------------------------------------
//                 Lazarus 2.0.10 - by BREAKOUTBOX
//                      latest changes 2021-04-02
// -----------------------------------------------------------------------------


interface

uses
  ctypes;


const
  LIB_SAMPLERATE = 'libsamplerate.dll';

const
  //  samplerate converter types:
  //  The following const values can be used to set the interpolator type
  //  using the function src_set_converter().
  SRC_SINC_BEST_QUALITY   = 0;
  SRC_SINC_MEDIUM_QUALITY = 1;
  SRC_SINC_FASTEST        = 2;
  SRC_ZERO_ORDER_HOLD     = 3;
  SRC_LINEAR              = 4;


type
  ppcfloat = ^pcfloat;  // This is the original translation from the C header file.
                        // => Compare this with libsamplerate headers Delphi translation
                        //    by Andrei Borovsky <anb@symmetrica.net> !

type
  // ToDo => does this make more sense ?
  FloatArray   = array[0..0] of Single;
  PFLOATARRAY  = ^FloatArray;
  PPFLOATARRAY = ^PFLOATARRAY;
  ShortArray   =  array[0..0] of SmallInt;
  PSHORTARRAY  = ^ShortArray;


type
  // Opaque data type SRC_STATE.
  PSRC_STATE = Pointer;

type
  // SRC_DATA is used to pass data to src_simple() and src_process().
  PSRC_DATA = ^TSRC_DATA;
  TSRC_DATA = record
    data_in           : pcfloat;
    data_out          : pcfloat;
    input_frames      : clong;
    output_frames     : clong;
    input_frames_used : clong;
    output_frames_gen : clong;
    end_of_input      : cint;
    src_ratio         : cdouble;
  end;



type
  // User supplied callback function type for use with src_callback_new()
  // and src_callback_read(). First parameter is the same pointer that was
  // passed into src_callback_new(). Second parameter is pointer to a
  // pointer. The user supplied callback function must modify *data to
  // point to the start of the user supplied float array. The user supplied
  // function must return the number of frames that **data points to.
  TSrcCallback = function( cb_data: pointer; data: ppcfloat): clong; cdecl;


//  Standard initialisation function: return an anonymous pointer to the
//  internal state of the converter. Choose a converter from the enums below.
//  Error returned in *error.
function src_new( converter_type: cint; channels: cint;
                  error: pcint): PSRC_STATE; cdecl external LIB_SAMPLERATE;


//  Initilisation for callback based API: return an anonymous pointer to the
//  internal state of the converter. Choose a converter from the enums below.
//  The cb_data pointer can point to any data or be set to NULL. Whatever the
//  value, when processing, user supplied function "func" gets called with
//  cb_data as first parameter.
function src_callback_new( func: TSrcCallback; converter_type: cint;
                           channels: cint; error: pcint;
                           cb_data: pointer): PSRC_STATE; cdecl external LIB_SAMPLERATE;


//  Cleanup all internal allocations.
//  Always returns NULL.
function src_delete( state: PSRC_STATE): PSRC_STATE; cdecl external LIB_SAMPLERATE;


//  Standard processing function.
//  Returns non zero on error.
function src_process( state: PSRC_STATE; data: PSRC_DATA): cint; cdecl external LIB_SAMPLERATE;


//  Callback based processing function. Read up to frames worth of data from
//  the converter int *data and return frames read or -1 on error.  *)
function src_callback_read( state: PSRC_STATE; src_ratio: cdouble; frames:
                            clong; data: pcfloat): clong; cdecl external LIB_SAMPLERATE;


//  Simple interface for performing a single conversion from input buffer to
//  output buffer at a fixed conversion ratio.
//  Simple interface does not require initialisation as it can only operate on
//  a single buffer worth of audio.
//
//  Important Note: The simple API is not designed to work on small chunks
//  of a larger piece of audio. If you attempt to use it this way you are
//  doing it wrong and will not get the results you want. For processing audio data
//  in chunks you must use the full api or the callback based api.
function src_simple( data: PSRC_DATA; converter_type: cint;
                     channels: cint): cint; cdecl external LIB_SAMPLERATE;


//  This library contains a number of different sample rate converters,
//  numbered 0 through N.
//
//  Here, returns a string giving either a name or a more full description of each
//  sample rate converter or NULL if no sample rate converter exists for
//  the given value. The converters are sequentially numbered from 0 to N.
function src_getname( converter_type: cint):String;
function src_getdescription( converter_type: cint):String;
function src_getversion():String;


//  Set a new SRC ratio. This allows step responses in the conversion ratio.
//  Returns non zero on error.
function src_set_ratio( state: PSRC_STATE; new_ratio: cdouble): cint; cdecl external LIB_SAMPLERATE;


//  Get the current channel count.
//  Returns negative on error, positive channel count otherwise
function src_get_channels( state: PSRC_STATE): cint; cdecl external LIB_SAMPLERATE;


//  Reset the internal SRC state.
//  Does not modify the quality settings.
//  Does not free any memory allocations.
//  Returns non zero on error.
function src_reset( state: PSRC_STATE):integer; {cint;} cdecl external LIB_SAMPLERATE;


//  Return TRUE if ratio is a valid conversion ratio, FALSE otherwise.
//  true is = 1,  false is = 0
function src_is_validratio( ratio: cdouble): boolean;

//  Return an error number.
function src_error( state: PSRC_STATE): cint; cdecl external LIB_SAMPLERATE;


//  Convert the error number into a string.
function src_strerror( error: cint): pchar; cdecl external LIB_SAMPLERATE;


// -----------------------------------------------------------------------------
// Extra helper functions for converting from short to float and back again.
// -----------------------------------------------------------------------------
procedure src_short_to_float_array( const input: pcshort; output: pcfloat;
                                    len: cint); cdecl external LIB_SAMPLERATE;
procedure src_float_to_short_array( const input: pcfloat; output: pcshort;
                                    len: cint); cdecl external LIB_SAMPLERATE;

procedure src_int_to_float_array( const input: pcint; output: pcfloat;
                                  len: cint); cdecl external LIB_SAMPLERATE;
procedure src_float_to_int_array( const input: pcfloat; output: pcint;
                                  len: cint); cdecl external LIB_SAMPLERATE;


// -----------------------------------------------------------------------------
// --- Lazarus helper functions by BREAKOUBOX 2021
// -----------------------------------------------------------------------------

// --- ConvertSampleRate_Float32() --- Result:= data.output_frames_gen ---------
// This function - on success - returns a newly allocated Mem on pointer pWav
function ConvertSampleRate_Float32( var pWavFloat:pcfloat;
                                    const SourceSR, TargetSR:ctypes.cint;
                                    const channels:dword;
                                    const input_frames:ctypes.clong;
                                    const converter_type:ctypes.cint  // for example = SRC_SINC_BEST_QUALITY;
                                    ):ctypes.clong;


implementation

uses
  Dialogs, SysUtils,  // for DEBUG .. 2021-03-05
  LConvEncoding;


// declare "hidden" libsamplerate DLL functions
function src_get_name( converter_type: cint): pchar; cdecl external LIB_SAMPLERATE;
function src_get_description( converter_type: cint): pchar; cdecl external LIB_SAMPLERATE;
function src_get_version():PChar; cdecl external LIB_SAMPLERATE;
function src_is_valid_ratio( ratio: cdouble): cint; cdecl external LIB_SAMPLERATE;


// make these function results PASCAL compatible
function src_getname( converter_type: cint):String;
begin
  Result:= CP1252ToUTF8( src_get_name( converter_type));
end;

function src_getdescription( converter_type: cint):String;
begin
  Result:= CP1252ToUTF8( src_get_description( converter_type));
end;

function src_getversion():String;
begin
  Result:= CP1252ToUTF8( src_get_version);
end;


//  Return TRUE if ratio is a valid conversion ratio, FALSE otherwise.
function src_is_validratio( ratio: cdouble): boolean;
begin
  Result:= boolean( src_is_valid_ratio( ratio));
  // true is <> 0,  false is = 0
end;


// -----------------------------------------------------------------------------
// --- ConvertSampleRate_Float32() --- Result:= data.output_frames_gen ---------
// This function - on success - returns a newly allocated Mem on pointer pWav
function ConvertSampleRate_Float32( var pWavFloat:pcfloat;
                                    const SourceSR, TargetSR:ctypes.cint;
                                    const channels:dword;
                                    const input_frames:ctypes.clong;
                                    const converter_type:ctypes.cint  // for example = SRC_SINC_BEST_QUALITY;
                                    ):ctypes.clong;
var
  data           : TSRC_DATA;
  pWavOutFloat   : pcfloat;
  BufferSize     : Int64;
  res            : cint;
begin
  Result:= 0;

  // check on valid SampleRate ..
  if (SourceSR < 11025) or (SourceSR > 192000) then  // Bugfix added 2021-04-02
    begin
      Result:= -1;
      Exit;
    end;

  //----------------------------------------------------------------------
  // resample to Target SampleRate
  // http://www.mega-nerd.com/SRC/api_misc.html#SRC_DATA
  //----------------------------------------------------------------------
  // The fields of this struct which MUST be filled in by the caller are:
  //      data_in       : A pointer to the input data samples.
  //      data_out      : A pointer to the output data samples.
  //      input_frames  : The number of frames of data pointed to by data_in.
  //      output_frames : Maximum number of frames pointer to by data_out.
  //      src_ratio     : Equal to output_sample_rate / input_sample_rate.

  // The input_frames and output_frames fields supply the converter with the lengths
  // of the arrays (in frames) pointed to by the data_in and data_out pointers
  // respectively. For monophinc data, these values would indicate the length
  // of the arrays while for multi channel data these values would be equal
  // to the the length of the array divided by the number of channels.
  //
  data.src_ratio:= TargetSR / SourceSR; // = for example (44100 / 11025);

  data.input_frames  := input_frames;
  data.output_frames := Round( data.input_frames * data.src_ratio);


  // The data_in pointer is used to pass audio data into the converter
  // while the data_out pointer supplies the converter with an array
  // to hold the converter's output. For a converter which has been configured
  // for multichannel operation, these pointers need to point to a single array of interleaved data.
  //
  BufferSize:= Round( data.src_ratio * data.input_frames * SizeOf( cfloat) * channels);
  GetMem( pWavOutFloat, BufferSize);
  //FillChar( pWavOutFloat^, BufferSize, 0);

  data.data_in  := @pWavFloat^;
  data.data_out := @pWavOutFloat^;

  // Finally, the src_ratio field specifies the conversion ratio defined as the
  // input sample rate divided by the output sample rate. For a connected set
  // of buffers, this value can be varies on each call to src_process resulting
  // in a time varying sample rate conversion process. For time varying
  // sample rate conversions, the ratio will be linearly interpolated
  // between the src_ratio value of the previous call to src_process
  // and the value for the current call.
  //
  if src_is_validratio( data.src_ratio)
    then
      begin
        //ShowMessage( 'src_is_validratio = TRUE   Ratio = ' +IntToStr( Round( data.src_ratio)))

        // When the src_simple function returns, output_frames_gen will be set to the
        // number of output frames generated, and input_frames_used will be set to the
        // number of input frames used to generate the provided number of output frames.
        //
        res:= src_simple( @data, converter_type, channels);   // result is  cint  !
        if res = 0
          then
            begin
              Result:= data.output_frames_gen;
              //ShowMessage( 'output_frames       ' +IntToStr( data.output_frames) +#13
              //             +'output_frames_gen  ' +IntToStr( data.output_frames_gen) );

              // --- free old Memory and assign Memory to fn VAR pointer ---
              FreeMem( pWavFloat);
              pWavFloat:= pWavOutFloat;  // swap pointers
            end
          else
            begin
              Result:= 0;
            end;
      end
    // on error  =>  because of  src_is_validratio = FALSE
    else Result:= 0;
end;


end.
