unit MainUnit;

{$mode objfpc}{$H+}

{-------------------------------------------------------------------------------
-                      Demo Project  Libsamplerate Test
-
-                             created:  2021-02-11
-                      latest changes:  2021-04-02
--------------------------------------------------------------------------------
-  0.1  created to test  libsamplerate.dll  (0.1.8 included here)
-  0.2  added src_simple() and two WaveForm images
-  0.3  added a different libsamplerate.dll,
-       the first one crashed with src_float_to_short_array()
-  0.4  bugfix in libsamplerate.pas - function ConvertSampleRate_Float32()
-------------------------------------------------------------------------------}

interface

uses
  Interfaces,
  Classes, SysUtils, Forms, Controls, Graphics, Dialogs, ExtCtrls, StdCtrls;


const
  VersionStr = '0.4  2021-04-02';


type
  { TForm1 }
  TForm1 = class(TForm)
    Button_src_simple: TButton;
    ButtonLibVersion: TButton;
    ButtonClose: TButton;
    ImageIn: TImage;
    ImageOut: TImage;
    Labelin: TLabel;
    LabelOut: TLabel;
    Memo1: TMemo;
    Panel1: TPanel;
    Panel2: TPanel;
    procedure ButtonCloseClick(Sender: TObject);
    procedure ButtonLibVersionClick(Sender: TObject);
    procedure Button_src_simpleClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
  private
    //
  public
    //
  end;

var
  Form1: TForm1;


implementation

{$R *.lfm}

uses
  ctypes,
  libsamplerate;


{ --- TForm1 ----------------------------------------------------------------- }
procedure TForm1.FormCreate(Sender: TObject);
begin
  Self.Caption:= Self.Caption +' Version ' +VersionStr;
end;


{ --- Buttons ---------------------------------------------------------------- }
procedure TForm1.ButtonCloseClick(Sender: TObject);
begin
  Close;
end;


procedure TForm1.ButtonLibVersionClick(Sender: TObject);
begin
  Memo1.Clear;
  Memo1.Append( 'Version Number of current DLL is = ' +src_getversion);
  Memo1.Append( '[ Latest known source code version is = 0.2.1 (2021-01-23) ]');
  Memo1.Append( '');
end;


procedure TForm1.Button_src_simpleClick(Sender: TObject);
var
  data           : TSRC_DATA;  // ... and NOT = PSRC_DATA because SRC_DATA in Lazarus IS already a pointer !
  converter_type : cint;       // = longint in case of Win32
  channels       : cint;       // = longint in case of Win32
  res            : cint;
  pWavIn         : pcfloat;
  pWavOut        : pcfloat;
  pTemp          : pcfloat;
  i              : integer;
  damper         : cfloat;     // added 2021-02-12
  b              : boolean;    // added 2021-02-12
const BufFramesIn = 128;
begin
  // http://www.mega-nerd.com/SRC/api_misc.html#SRC_DATA
  //
  // The fields of this struct which MUST be filled in by the caller are:
  //      data_in       : A pointer to the input data samples.
  //      data_out      : A pointer to the output data samples.
  //      input_frames  : The number of frames of data pointed to by data_in.
  //      output_frames : Maximum number of frames pointer to by data_out.
  //      src_ratio     : Equal to output_sample_rate / input_sample_rate.


  // The data_in pointer is used to pass audio data into the converter
  // while the data_out pointer supplies the converter with an array
  // to hold the converter's output. For a converter which has been configured
  // for multichannel operation, these pointers need to point to a single array of interleaved data.
  GetMem( pWavIn,    BufFramesIn * SizeOf( cfloat));
  GetMem( pWavOut, 4*BufFramesIn * SizeOf( cfloat));

  // -------------------------------------------------------------------------
  // randomly fill the Input Buffer (instead of opening a REAL audio file ...)
  // -------------------------------------------------------------------------
  pTemp:= pWavIn; // warning: float, not integer !
  damper:= 0;
  for i:= 0 to BufFramesIn -1 do
    begin
      pTemp^:= (damper +(Random( 65536)-32768) / 32768) / 2;
      damper:= pTemp^;
      inc( pTemp);
    end;
  data.data_in  := pWavIn;
  data.data_out := pWavOut;


  // The input_frames and output_frames fields supply the converter with the lengths
  // of the arrays (in frames) pointed to by the data_in and data_out pointers
  // respectively. For monophinc data, these values would indicate the length
  // of the arrays while for multi channel data these values would be equal
  // to the the length of the array divided by the number of channels.
  //
  data.input_frames  := BufFramesIn;
  data.output_frames := 4*BufFramesIn;


  // Finally, the src_ratio field specifies the conversion ratio defined as the
  // input sample rate divided by the output sample rate. For a connected set
  // of buffers, this value can be varies on each call to src_process resulting
  // in a time varying sample rate conversion process. For time varying
  // sample rate conversions, the ratio will be linearly interpolated
  // between the src_ratio value of the previous call to src_process
  // and the value for the current call.
  //
  data.src_ratio:= 4;  // = for example (44100 / 11025);
  b:= src_is_validratio( data.src_ratio);
  if b
    then Memo1.Append( 'src_is_validratio = TRUE')
    else Memo1.Append( 'src_is_validratio = FALSE');


  // When the src_simple function returns, output_frames_gen will be set to the
  // number of output frames generated and input_frames_used will be set to the
  // number of input frames used to generate the provided number of output frames.
  //
  converter_type:= SRC_SINC_BEST_QUALITY;
  channels:= 1;
  res:= src_simple( @data, converter_type, channels);   // result is  cint  !
  if res = 0
    then Memo1.Append( 'src_simple() result:    ' +IntToStr( res) +' = ' +ShortString( src_strerror( res)))
    else ShowMessage( 'src_simple() result:    ' +IntToStr( res) +#13#13
                      +'Resampling error:    ' +ShortString( src_strerror( res)));


  // ***************************************************************************
  // fill a TImage with a WaveForm graphic:
  with ImageIn do
    begin
      pTemp:= pWavIn; // warning: float !

      Width:= 4 * BufFramesIn;

      Picture.Bitmap.Width:= Width;
      Picture.Bitmap.Height:= ImageIn.Height;
      Picture.Bitmap.Canvas.Brush.Color:= $00000;
      Picture.Bitmap.Canvas.FillRect( 0, 0,
                                      Picture.Bitmap.Width,
                                      Picture.Bitmap.Height);

      // draw a horizontal line
      Picture.Bitmap.Canvas.Pen.Color:= $004400;
      Picture.Bitmap.Canvas.MoveTo( 0, Picture.Bitmap.Height div 2);
      Picture.Bitmap.Canvas.LineTo( width, Picture.Bitmap.Height div 2);

      // draw the WaveForm
      Picture.Bitmap.Canvas.Pen.Color:= $00FF00;
      Picture.Bitmap.Canvas.MoveTo( 0, Picture.Bitmap.Height div 2);
      for i:= 1 to BufFramesIn do
        begin
          //ShowMessage( IntToStr( Round( pTemp^) * Picture.Bitmap.Height) );
          Picture.Bitmap.Canvas.LineTo( 4*i,
                                        (Picture.Bitmap.Height div 2)
                                        +Round( 0.9 * pTemp^ *(Picture.Bitmap.Height div 2) ));
          inc( pTemp);
        end;
    end;

  // ***************************************************************************
  // fill a TImage with a WaveForm graphic:
  if src_is_validratio( data.src_ratio) then
  with ImageOut do
    begin
      pTemp:= pWavOut; // warning: float !

      Width:= 4 * BufFramesIn;

      Picture.Bitmap.Width:= Width;
      Picture.Bitmap.Height:= ImageIn.Height;
      Picture.Bitmap.Canvas.Brush.Color:= $00000;
      Picture.Bitmap.Canvas.FillRect( 0, 0,
                                      Picture.Bitmap.Width,
                                      Picture.Bitmap.Height);

      // draw a horizontal line
      Picture.Bitmap.Canvas.Pen.Color:= $004400;
      Picture.Bitmap.Canvas.MoveTo( 0, Picture.Bitmap.Height div 2);
      Picture.Bitmap.Canvas.LineTo( width, Picture.Bitmap.Height div 2);

      // draw the WaveForm
      Picture.Bitmap.Canvas.Pen.Color:= $00FF00;
      Picture.Bitmap.Canvas.MoveTo( 0, Picture.Bitmap.Height div 2);
      for i:= 1 to 4*BufFramesIn do
        begin
          Picture.Bitmap.Canvas.LineTo( i,
                                        (Picture.Bitmap.Height div 2)
                                        +Round( 0.9 * pTemp^ *(Picture.Bitmap.Height div 2) ));
          inc( pTemp);
        end;
    end;

  // finally clean up :
  FreeMem( pWavIn);
  FreeMem( pWavOut);
end;


end.

